<?php
session_start();

// Roles allowed
if (!isset($_SESSION['role']) || !in_array($_SESSION['role'], ['staff_user','branch_head','letter_admin','super_admin'])) {
    header('Location: unauthorized.php'); exit();
}

require_once 'db_connection.php'; // mysqli $conn with utf8mb4

$role       = $_SESSION['role'];
$user_id    = (int)($_SESSION['user_id'] ?? 0);
$branch_id  = (int)($_SESSION['branch_id'] ?? 0);

/**
 * Letters with NO action and assigned 3–7 days ago.
 * Use SQL-side calc:
 *   DATEDIFF(NOW(), l.assign_date) BETWEEN 3 AND 7
 */
$conds  = ["l.action = 'no'", "l.assign_date IS NOT NULL", "DATEDIFF(NOW(), l.assign_date) BETWEEN 3 AND 7"];
$params = [];
$types  = "";

// Role scoping
if ($role === 'branch_head') {
    $conds[] = "l.branch_id = ?";
    $params[] = $branch_id; $types .= "i";
} elseif ($role === 'staff_user') {
    $conds[] = "l.forward_to = ?";
    $params[] = $user_id;   $types .= "i";
}
// letter_admin/super_admin see all

$where = "WHERE ".implode(" AND ", $conds);

$sql = "
SELECT
  l.letter_id,
  l.letter_number,
  l.letter_date,
  l.department,
  l.subject,
  l.assign_date,
  DATEDIFF(NOW(), l.assign_date) AS days_since_assign,
  b.branch_name
FROM letters l
JOIN branches b ON l.branch_id = b.branch_id
{$where}
ORDER BY l.assign_date DESC, l.letter_id DESC
";

$stmt = $conn->prepare($sql);
if ($types !== "") { $stmt->bind_param($types, ...$params); }
$stmt->execute();
$res = $stmt->get_result();
$stmt->close();
// $conn->close();
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8"/>
  <meta name="viewport" content="width=device-width, initial-scale=1"/>
  <title>Letters — No Action for 3–7 Days</title>

  <!-- Bootstrap 5 -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet"
        integrity="sha384-QWTKZyjpPEjISv5WaRU9OFeRpok6YctnYmDr5pNlyT2bRjXh0JMhjY6hW+ALEwIH" crossorigin="anonymous">
  <!-- DataTables (Bootstrap 5 theme) -->
  <link rel="stylesheet" href="https://cdn.datatables.net/1.13.6/css/dataTables.bootstrap5.min.css">
  <!-- Icons -->
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css"/>

  <style>
    :root { --brand:#0d6efd; --bg:#f6f8fb; }
    body { background: var(--bg); padding-bottom: 40px; }
    .page-head{
      background: linear-gradient(135deg, var(--brand), #5aa2ff);
      color:#fff; border-radius:1rem; padding:1rem 1.25rem;
      box-shadow:0 6px 24px rgba(13,110,253,.25); margin:1rem;
    }
    .table thead th { white-space: nowrap; }
    .row-warn { background:#fff4e5; } /* subtle orange for day 7 */
    .badge-soft {
      background: rgba(13,110,253,.1);
      border:1px solid rgba(13,110,253,.2);
      color:#0b1320;
    }
  </style>
</head>
<body>
<div class="container-fluid">

  <!-- Header -->
  <div class="page-head d-flex align-items-center justify-content-between">
    <div class="d-flex align-items-center gap-3">
      <i class="fa-solid fa-hourglass-half fs-3"></i>
      <div>
        <h5 class="mb-0 fw-bold">Letters Not Actioned for 3–7 Days</h5>
        <small>Letters with <code>action = 'no'</code> where <code>DATEDIFF(NOW(), assign_date)</code> is 3–7 days</small>
      </div>
    </div>
    <div>
      <?php
        if     ($role === 'super_admin')    $back = 'super_admin_dashboard.php';
        elseif ($role === 'letter_admin')   $back = 'letter_admin_dashboard.php';
        elseif ($role === 'branch_head')    $back = 'branch_head_dashboard.php';
        else                                $back = 'staff_user_dashboard.php';
      ?>
      <a class="btn btn-light" href="<?= $back; ?>"><i class="fa-solid fa-arrow-left-long me-1"></i> Dashboard</a>
    </div>
  </div>

  <!-- Table -->
  <div class="card shadow-sm m-3">
    <div class="table-responsive">
      <table id="lettersTable" class="table table-striped align-middle mb-0">
        <thead class="table-light">
        <tr>
          <th>Letter No.</th>
          <th>Letter Date</th>
          <th>Department</th>
          <th>Subject</th>
          <th>Assign Date</th>
          <th>Days Since Assign</th>
          <th>Branch</th>
        </tr>
        </thead>
        <tbody>
        <?php if ($res && $res->num_rows > 0): ?>
          <?php while ($r = $res->fetch_assoc()): ?>
            <?php
              $days = (int)($r['days_since_assign'] ?? 0);
              $rowClass = ($days >= 7) ? 'row-warn' : '';
            ?>
            <tr class="<?= $rowClass; ?>">
              <td><?= htmlspecialchars($r['letter_number'] ?? '-', ENT_QUOTES, 'UTF-8'); ?></td>
              <td><?= htmlspecialchars($r['letter_date'] ?? '-', ENT_QUOTES, 'UTF-8'); ?></td>
              <td><?= htmlspecialchars($r['department'] ?? '-', ENT_QUOTES, 'UTF-8'); ?></td>
              <td><?= htmlspecialchars($r['subject'] ?? '-', ENT_QUOTES, 'UTF-8'); ?></td>
              <td><?= htmlspecialchars($r['assign_date'] ?? '-', ENT_QUOTES, 'UTF-8'); ?></td>
              <td>
                <span class="badge rounded-pill <?= $days >= 7 ? 'text-bg-warning' : 'badge-soft'; ?>">
                  <?= $days; ?> day<?= $days === 1 ? '' : 's'; ?>
                </span>
              </td>
              <td><?= htmlspecialchars($r['branch_name'] ?? '-', ENT_QUOTES, 'UTF-8'); ?></td>
            </tr>
          <?php endwhile; ?>
        <?php endif; // IMPORTANT: no else row with colspan ?>
        </tbody>
      </table>
    </div>
  </div>

</div>

<!-- JS: jQuery + DataTables + Bootstrap 5 theme -->
<script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
<script src="https://cdn.datatables.net/1.13.6/js/jquery.dataTables.min.js"></script>
<script src="https://cdn.datatables.net/1.13.6/js/dataTables.bootstrap5.min.js"></script>
<script>
  $(function(){
    $('#lettersTable').DataTable({
      pageLength: 25,
      order: [[4, 'desc']], // Assign Date desc
      language: {
        emptyTable: "No letters have been left unactioned for 3–7 days."
      }
    });
  });
</script>
</body>
</html>
