<?php
session_start();
require_once 'db_connection.php';

// Require login
if (!isset($_SESSION['user_id'], $_SESSION['role'])) {
    header("Location: login.php"); exit;
}

$user_id   = (int)$_SESSION['user_id'];
$role      = $_SESSION['role'];
$branch_id = (int)($_SESSION['branch_id'] ?? 0);

// ----- Build secure role-scoped query -----
$conds  = [];
$params = [];
$types  = "";

// Role access:
// super_admin & letter_admin => all
if ($role === 'branch_head') {
    $conds[] = "fm.branch_id = ?";
    $params[] = $branch_id; $types .= "i";
} elseif ($role === 'staff_user') {
    $conds[] = "fm.user_id = ?";
    $params[] = $user_id;   $types .= "i";
}
// letter_admin and super_admin see all records

$where = $conds ? ("WHERE ".implode(" AND ", $conds)) : "";

$sql = "
SELECT
  fm.file_id,
  fm.file_name,
  fm.file_number,
  fm.subject,
  fm.status,
  b.branch_name,
  u.username AS user_name
FROM file_management fm
JOIN branches b ON fm.branch_id = b.branch_id
JOIN users    u ON fm.user_id   = u.user_id
{$where}
ORDER BY fm.file_id DESC
";

$stmt = $conn->prepare($sql);
if ($types !== "") { $stmt->bind_param($types, ...$params); }
$stmt->execute();
$result = $stmt->get_result();
$stmt->close();

// Back link
if     ($role === 'super_admin')   $back_url = 'super_admin_dashboard.php';
elseif ($role === 'letter_admin')  $back_url = 'letter_admin_dashboard.php';
elseif ($role === 'branch_head')   $back_url = 'branch_head_dashboard.php';
else                               $back_url = 'staff_user_dashboard.php';
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8"/>
  <meta name="viewport" content="width=device-width, initial-scale=1"/>
  <title>File Management View</title>

  <!-- Bootstrap 5 -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css"
        rel="stylesheet"
        integrity="sha384-QWTKZyjpPEjISv5WaRU9OFeRpok6YctnYmDr5pNlyT2bRjXh0JMhjY6hW+ALEwIH"
        crossorigin="anonymous"/>

  <!-- DataTables (Bootstrap 5 theme) -->
  <link rel="stylesheet" href="https://cdn.datatables.net/1.13.6/css/dataTables.bootstrap5.min.css"/>

  <!-- Icons -->
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css"/>

  <style>
    :root { --bg:#f6f8fb; }
    body { background: var(--bg); }
    .wrap { max-width: 1200px; }
    .page-head{
      background: linear-gradient(135deg, #0d6efd, #5aa2ff);
      color:#fff; border-radius:1rem; padding:1rem 1.25rem;
      box-shadow:0 6px 24px rgba(13,110,253,.25); margin: 1.25rem 0;
    }
    .card { border: none; border-radius: 1rem; box-shadow: 0 10px 30px rgba(0,0,0,.06); }
    .badge-soft {
      background: rgba(13,110,253,.08);
      border:1px solid rgba(13,110,253,.15);
      color:#0b1320;
    }
  </style>
</head>
<body>
<div class="container wrap py-3">

  <!-- Header -->
  <div class="page-head d-flex align-items-center justify-content-between">
    <div class="d-flex align-items-center gap-3">
      <i class="fa-solid fa-folder-open fs-3"></i>
      <div>
        <h5 class="mb-0 fw-bold">File Management Records</h5>
        <small class="text-white-75">
          <?php if (in_array($role, ['super_admin','letter_admin'])): ?>
            Viewing: <strong>All Branches</strong>
          <?php elseif ($role === 'branch_head'): ?>
            Viewing: <strong>Your Branch</strong>
          <?php else: ?>
            Viewing: <strong>Your Files</strong>
          <?php endif; ?>
        </small>
      </div>
    </div>
    <a href="<?= htmlspecialchars($back_url, ENT_QUOTES, 'UTF-8'); ?>" class="btn btn-light">
      <i class="fa-solid fa-arrow-left-long me-1"></i> Back to Dashboard
    </a>
  </div>

  <!-- Table -->
  <div class="card p-3">
    <div class="table-responsive">
      <table id="fileTable" class="table table-striped align-middle mb-0">
        <thead class="table-light">
          <tr>
            <th>File Name</th>
            <th>File Number</th>
            <th>Subject</th>
            <th>Branch</th>
            <th>User</th>
            <th>Status</th>
            <th style="width: 160px;">Action</th>
          </tr>
        </thead>
        <tbody>
          <?php if ($result && $result->num_rows > 0): ?>
            <?php while ($row = $result->fetch_assoc()): ?>
              <?php
                $file_id    = (int)$row['file_id'];
                $file_name  = htmlspecialchars($row['file_name']   ?? '-', ENT_QUOTES, 'UTF-8');
                $file_no    = htmlspecialchars($row['file_number'] ?? '-', ENT_QUOTES, 'UTF-8');
                $subject    = htmlspecialchars($row['subject']     ?? '-', ENT_QUOTES, 'UTF-8');
                $branchName = htmlspecialchars($row['branch_name'] ?? '-', ENT_QUOTES, 'UTF-8');
                $userName   = htmlspecialchars($row['user_name']   ?? '-', ENT_QUOTES, 'UTF-8');
                $status     = htmlspecialchars($row['status']      ?? '-', ENT_QUOTES, 'UTF-8');

                // Status styling
                $statusClass = 'badge-soft';
                if ($status === 'Active')        $statusClass = 'text-bg-success';
                elseif ($status === 'Archived')  $statusClass = 'text-bg-secondary';
                elseif ($status === 'Under Review') $statusClass = 'text-bg-warning';
              ?>
              <tr>
                <td><?= $file_name; ?></td>
                <td><?= $file_no; ?></td>
                <td><?= $subject; ?></td>
                <td><?= $branchName; ?></td>
                <td><?= $userName; ?></td>
                <td><span class="badge rounded-pill <?= $statusClass; ?>"><?= $status; ?></span></td>
                <td>
                  <div class="d-flex gap-2">
                    <a href="file_edit.php?file_id=<?= $file_id; ?>" class="btn btn-sm btn-primary">
                      <i class="fa-solid fa-pen-to-square me-1"></i>Edit
                    </a>
                    <button type="button"
                            class="btn btn-sm btn-outline-secondary"
                            data-bs-toggle="modal"
                            data-bs-target="#fileModal"
                            data-file-id="<?= $file_id; ?>">
                      <i class="fa-solid fa-eye me-1"></i>View
                    </button>
                  </div>
                </td>
              </tr>
            <?php endwhile; ?>
          <?php endif; // no else <tr> so DataTables can show its own message ?>
        </tbody>
      </table>
    </div>
  </div>

</div>

<!-- Modal -->
<div class="modal fade" id="fileModal" tabindex="-1" aria-labelledby="fileModalLabel" aria-hidden="true">
  <div class="modal-dialog modal-lg modal-dialog-scrollable">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="fileModalLabel">File Details</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body" id="fileDetails">
        <div class="text-center text-muted py-3">Loading…</div>
      </div>
    </div>
  </div>
</div>

<!-- JS: Bootstrap + jQuery + DataTables (BS5) -->
<script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
<script src="https://cdn.datatables.net/1.13.6/js/jquery.dataTables.min.js"></script>
<script src="https://cdn.datatables.net/1.13.6/js/dataTables.bootstrap5.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"
        integrity="sha384-YvpcrYf0tY3lHB60NNkmXc5s9fDVZLESaAA55NDzOxhy9GkcIdslK1eN7N6jIeHz"
        crossorigin="anonymous"></script>
<script>
  // DataTable
  $(function(){
    $('#fileTable').DataTable({
      pageLength: 25,
      order: [[0, 'asc']],
      language: {
        emptyTable: "No file records found."
      }
    });
  });

  // Modal: load details via AJAX
  const fileModal = document.getElementById('fileModal');
  fileModal.addEventListener('show.bs.modal', event => {
    const button = event.relatedTarget;
    const fileId = button?.getAttribute('data-file-id');
    const target = document.getElementById('fileDetails');
    target.innerHTML = '<div class="text-center text-muted py-3">Loading…</div>';
    if (!fileId) return;

    // Fetch details
    fetch('get_file_details.php?file_id=' + encodeURIComponent(fileId), { credentials: 'same-origin' })
      .then(r => r.text())
      .then(html => target.innerHTML = html)
      .catch(() => target.innerHTML = '<div class="text-danger">Failed to load details.</div>');
  });
</script>
</body>
</html>
