<?php
session_start();
require_once 'db_connection.php'; // must provide $conn (mysqli) and set utf8mb4

// If already logged in, send to dashboard (optional)
if (!empty($_SESSION['role'])) {
  switch ($_SESSION['role']) {
    case 'branch_head':   header('Location: branch_head_dashboard.php'); exit;
    case 'letter_admin':  header('Location: letter_admin_dashboard.php'); exit;
    case 'staff_user':    header('Location: staff_user_dashboard.php'); exit;
    case 'super_admin':   header('Location: super_admin_dashboard.php'); exit;
    default:              header('Location: unauthorized.php'); exit;
  }
}

$error_message = '';

// CSRF helper
if (empty($_SESSION['csrf_login'])) {
  $_SESSION['csrf_login'] = bin2hex(random_bytes(32));
}
$csrf_token = $_SESSION['csrf_login'];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // CSRF check
    if (!hash_equals($_SESSION['csrf_login'] ?? '', $_POST['csrf'] ?? '')) {
        $error_message = 'Session expired. Please refresh and try again.';
    } else {
        // Basic sanitization
        $username = trim($_POST['username'] ?? '');
        $password = $_POST['password'] ?? '';

        if ($username === '' || $password === '') {
            $error_message = 'Please enter both username and password.';
        } else {
            // Fetch by username
            $sql = "SELECT user_id, username, password, role, branch_id, status
                    FROM users
                    WHERE username = ? LIMIT 1";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param('s', $username);
            $stmt->execute();
            $res = $stmt->get_result();

            if ($res && $res->num_rows === 1) {
                $user = $res->fetch_assoc();

                if ($user['status'] !== 'active') {
                    $error_message = "Your account is inactive. Please contact the administrator.";
                } else {
                    $stored = (string)$user['password'];

                    // hashed?
                    $ok = password_verify($password, $stored);

                    // Plain-text fallback (remove once all are hashed)
                    if (!$ok && hash_equals($stored, $password)) {
                        $ok = true;
                    }

                    if ($ok) {
                        // Prevent session fixation
                        session_regenerate_id(true);

                        $_SESSION['user_id']   = (int)$user['user_id'];
                        $_SESSION['username']  = $user['username'];
                        $_SESSION['role']      = $user['role'];
                        $_SESSION['branch_id'] = (int)$user['branch_id'];

                        // redirect by role
                        switch ($user['role']) {
                            case 'branch_head':   header('Location: branch_head_dashboard.php'); exit;
                            case 'letter_admin':  header('Location: letter_admin_dashboard.php'); exit;
                            case 'staff_user':    header('Location: staff_user_dashboard.php'); exit;
                            case 'super_admin':   header('Location: super_admin_dashboard.php'); exit;
                            default:              header('Location: unauthorized.php'); exit;
                        }
                    } else {
                        $error_message = "Invalid username or password";
                    }
                }
            } else {
                $error_message = "User not found";
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title>Letter Tracking System – Login</title>

  <!-- Bootstrap 5 -->
  <link
    href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css"
    rel="stylesheet"
    integrity="sha384-QWTKZyjpPEjISv5WaRU9OFeRpok6YctnYmDr5pNlyT2bRjXh0JMhjY6hW+ALEwIH"
    crossorigin="anonymous"
  />
  <!-- Bootstrap Icons (for eye icon) -->
  <link
    href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.css"
    rel="stylesheet"
  />

  <style>
    :root{
      --brand:#0d6efd; --brand-2:#5aa2ff; --bg:#0b1020; --glass:#ffffff;
    }
    html, body { height:100%; }
    body{
      display:grid; place-items:center; min-height:100vh;
      background:
        radial-gradient(60rem 60rem at 120% -10%, rgba(13,110,253,.25), transparent 60%),
        radial-gradient(48rem 48rem at -10% 110%, rgba(90,162,255,.22), transparent 60%),
        linear-gradient(180deg,#0b1020 0%, #0e1530 100%);
      color:#0b1320;
    }

    /* Glass card */
    .login-wrap{
      width:min(95vw, 980px);
      backdrop-filter: blur(10px);
      background: linear-gradient(180deg, rgba(255,255,255,.85), rgba(255,255,255,.78));
      border:1px solid rgba(255,255,255,.35);
      border-radius: 1.25rem;
      box-shadow: 0 30px 80px rgba(0,0,0,.35), inset 0 1px 0 rgba(255,255,255,.25);
      overflow:hidden;
    }
    .hero{
      background: radial-gradient(60rem 60rem at 120% -10%, rgba(13,110,253,.25), transparent 60%),
                  linear-gradient(135deg, var(--brand), var(--brand-2));
      color:#fff;
      padding: 2rem 2rem;
    }
    @media (min-width: 768px){
      .grid{
        display:grid; grid-template-columns: 1.2fr 1fr;
      }
      .hero{ padding: 2.25rem 2.5rem; }
    }
    .brandmark{
      display:flex; align-items:center; gap:.9rem;
    }
    .brandmark img{
      height:56px; width:56px; object-fit:cover; border-radius: .9rem;
      border:2px solid rgba(255,255,255,.65);
      background:#fff;
    }
    .hero h1{ font-weight:800; letter-spacing:.2px; margin:.75rem 0 .25rem; }
    .hero .lead{ opacity:.95; }

    .login-panel{
      padding: 1.5rem;
      background: rgba(255,255,255,.35);
      backdrop-filter: blur(6px);
    }
    @media (min-width: 768px){
      .login-panel{ padding: 2rem; }
    }

    .card-inner{
      background:#fff; border-radius: 1rem; padding:1.25rem;
      box-shadow: 0 10px 30px rgba(13,110,253,.10);
    }
    @media (min-width: 768px){
      .card-inner{ padding:1.5rem; }
    }

    .form-control{
      padding: .85rem 1rem; border-radius:.8rem;
    }
    .input-group .form-control{ border-right:0; }
    .input-group .btn{
      border-left:0; border-radius:0 .8rem .8rem 0;
    }
    .btn-brand{
      padding:.9rem 1rem; border-radius:.8rem; font-weight:600;
    }
    .subtle{ color:#6c757d; }
    .support-links a{ text-decoration: none; }
    .support-links a:hover{ text-decoration: underline; }

    .footer-note{
      text-align:center; color:#cfd6e6; font-size:.9rem; margin-top:1rem;
    }
  </style>
</head>
<body>

  <div class="login-wrap grid">
    <!-- Left: brand/intro -->
    <section class="hero">
      <div class="brandmark">
        <img src="logo.jpg" alt="Logo" onerror="this.style.display='none'">
        <div>
          <div class="fw-semibold">Valvettithurai Urban Council</div>
          <small class="text-white-50">Official Portal</small>
        </div>
      </div>

      <h1>Letter Tracking System</h1>
      <p class="lead mb-4">Sign in to manage incoming letters, assignments, and branch activity efficiently.</p>

      <ul class="mb-0">
        <li>Secure role-based access</li>
        <li>Real-time KPIs and tracking</li>
        <li>Streamlined file management</li>
      </ul>
    </section>

    <!-- Right: form -->
    <section class="login-panel">
      <div class="card-inner">
        <?php if (!empty($error_message)): ?>
          <div class="alert alert-danger py-2 mb-3" role="alert">
            <?= htmlspecialchars($error_message, ENT_QUOTES, 'UTF-8'); ?>
          </div>
        <?php endif; ?>

        <form method="post" action="" novalidate>
          <input type="hidden" name="csrf" value="<?= htmlspecialchars($csrf_token, ENT_QUOTES, 'UTF-8'); ?>">

          <div class="mb-3">
            <label for="username" class="form-label">Username</label>
            <div class="input-group">
              <span class="input-group-text"><i class="bi bi-person"></i></span>
              <input id="username" name="username" type="text" class="form-control" required autocomplete="username" autofocus />
            </div>
          </div>

          <div class="mb-2">
            <label for="password" class="form-label d-flex justify-content-between">
              <span>Password</span>
              <a href="forgot_password.php" class="small">Forgot?</a>
            </label>
            <div class="input-group">
              <span class="input-group-text"><i class="bi bi-shield-lock"></i></span>
              <input id="password" name="password" type="password" class="form-control" required autocomplete="current-password" />
              <button class="btn btn-outline-secondary" type="button" id="togglePw" aria-label="Show password">
                <i class="bi bi-eye"></i>
              </button>
            </div>
          </div>

          <div class="d-grid mt-3">
            <button type="submit" class="btn btn-primary btn-brand" id="signInBtn">
              <span class="spinner-border spinner-border-sm me-2 d-none" id="btnSpinner" role="status" aria-hidden="true"></span>
              Sign in
            </button>
          </div>

          <div class="support-links text-center mt-3 subtle">
            Need access? <a href="contact_admin.php">Contact the administrator</a>
          </div>
        </form>
      </div>

      <div class="footer-note">© <?= date('Y'); ?> Valvettithurai Urban Council</div>
    </section>
  </div>

  <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"
          integrity="sha384-YvpcrYf0tY3lHB60NNkmXc5s9fDVZLESaAA55NDzOxhy9GkcIdslK1eN7N6jIeHz"
          crossorigin="anonymous"></script>
  <script>
    // Show/hide password
    (function(){
      const pw = document.getElementById('password');
      const btn = document.getElementById('togglePw');
      const icon = btn.querySelector('i');
      btn.addEventListener('click', function(){
        const isPass = pw.type === 'password';
        pw.type = isPass ? 'text' : 'password';
        icon.classList.toggle('bi-eye');
        icon.classList.toggle('bi-eye-slash');
        btn.setAttribute('aria-label', isPass ? 'Hide password' : 'Show password');
      });
    })();

    // Button loading state
    (function(){
      const form = document.querySelector('form');
      const btn = document.getElementById('signInBtn');
      const spinner = document.getElementById('btnSpinner');
      form.addEventListener('submit', function(){
        btn.setAttribute('disabled', 'disabled');
        spinner.classList.remove('d-none');
      });
    })();
  </script>
</body>
</html>
