<?php
// Start the session and include the database connection
session_start();
require_once 'db_connection.php';  // DB connection

// Auth: Staff User only
if (!isset($_SESSION['user_id']) || ($_SESSION['role'] ?? '') !== 'staff_user') {
    header("Location: login.php");
    exit();
}

$user_id = (int)($_SESSION['user_id'] ?? 0);

// Handle Seen Update
$success_message = $error_message = null;
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_seen'])) {
    $letter_id = (int)($_POST['letter_id'] ?? 0);
    $seen = (isset($_POST['seen']) && $_POST['seen'] == '1') ? 1 : 0;

    $sql = "UPDATE letters SET seen = ? WHERE letter_id = ? AND forward_to = ?";
    if ($stmt = $conn->prepare($sql)) {
        $stmt->bind_param("iii", $seen, $letter_id, $user_id);
        if ($stmt->execute()) {
            $success_message = "Letter status updated successfully.";
        } else {
            $error_message = "Error updating letter status: " . $stmt->error;
        }
        $stmt->close();
    } else {
        $error_message = "Failed to prepare statement.";
    }
}

// Fetch letters assigned to this staff user where seen = 0 (pending)
$letters = [];
$sqlList = "SELECT letter_id, letter_number, letter_date, department, subject, received_date, scan_copy, seen
            FROM letters
            WHERE forward_to = ? AND seen = 0
            ORDER BY letter_date DESC";
if ($stmt = $conn->prepare($sqlList)) {
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $res = $stmt->get_result();
    while ($row = $res->fetch_assoc()) { $letters[] = $row; }
    $stmt->close();
} else {
    $error_message = "Error fetching letters list.";
}

$conn->close();

function h($v){ return htmlspecialchars((string)$v, ENT_QUOTES, 'UTF-8'); }
?>
<!DOCTYPE html>
<html lang="en" data-bs-theme="light">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>Pending Letters • Staff User</title>

  <!-- Bootstrap 5.3 + Icons -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.css" rel="stylesheet">

  <style>
    :root{
      --brand: #155eef;
      --surface: #ffffff;
      --border: #e9eef7;
      --radius: 14px;
    }
    body{ background: #f6f8ff; }
    .page-wrap{ max-width: 1200px; }
    .page-card{
      background: var(--surface);
      border: 1px solid var(--border);
      border-radius: var(--radius);
    }
    .table thead th{
      position: sticky; top: 0; z-index: 1;
      background: #f8faff; border-bottom: 1px solid var(--border);
    }
    .badge-soft{
      background: #fff3f3; color:#b42318; border:1px solid #ffd5d5;
    }
    .btn-soft{
      background:#eef3ff; border-color:#dbe6ff; color:#155eef;
    }
    .btn-soft:hover{ background:#e3ebff; color:#0f4adf; }
    .scan-link{
      white-space: nowrap; overflow: hidden; text-overflow: ellipsis; max-width: 240px; display: inline-block;
    }
  </style>
</head>
<body>

  <header class="border-bottom bg-white">
    <div class="container page-wrap d-flex align-items-center justify-content-between py-3">
      <div class="d-flex align-items-center gap-3">
        <a href="staff_user_dashboard.php" class="btn btn-outline-secondary btn-sm"><i class="bi bi-arrow-left"></i></a>
        <div>
          <h5 class="mb-0">Pending Letters</h5>
          <small class="text-secondary">Items assigned to you and not yet seen</small>
        </div>
      </div>
      <img src="logo.jpg" alt="Logo" width="44" height="44" class="rounded">
    </div>
  </header>

  <main class="container page-wrap my-4">
    <!-- Alerts -->
    <?php if ($success_message): ?>
      <div class="alert alert-success d-flex align-items-center" role="alert">
        <i class="bi bi-check-circle-fill me-2"></i>
        <div><?= h($success_message) ?></div>
      </div>
    <?php endif; ?>
    <?php if ($error_message): ?>
      <div class="alert alert-danger d-flex align-items-center" role="alert">
        <i class="bi bi-exclamation-triangle-fill me-2"></i>
        <div><?= h($error_message) ?></div>
      </div>
    <?php endif; ?>

    <!-- Card -->
    <div class="page-card p-3 p-md-4">
      <div class="d-flex align-items-center justify-content-between mb-3">
        <div class="d-flex align-items-center gap-2">
          <span class="badge text-bg-primary"><i class="bi bi-envelope-paper me-1"></i> Pending</span>
          <span class="text-secondary small">Total: <strong><?= count($letters) ?></strong></span>
        </div>
        <div class="d-flex gap-2">
          <a href="staff_user_dashboard.php" class="btn btn-outline-secondary btn-sm"><i class="bi bi-grid me-1"></i> Dashboard</a>
          <a href="view_letters_statistics.php" class="btn btn-soft btn-sm"><i class="bi bi-pie-chart me-1"></i> Statistics</a>
        </div>
      </div>

      <?php if (!count($letters)): ?>
        <div class="text-center py-5">
          <i class="bi bi-inboxes fs-1 text-secondary d-block mb-2"></i>
          <h6 class="mb-1">No pending letters</h6>
          <p class="text-secondary mb-3">Everything assigned to you has been marked as <em>seen</em>.</p>
          <a href="staff_user_dashboard.php" class="btn btn-primary btn-sm"><i class="bi bi-arrow-left-short"></i> Back to Dashboard</a>
        </div>
      <?php else: ?>
        <div class="table-responsive">
          <table class="table table-hover align-middle mb-0">
            <thead>
              <tr>
                <th style="min-width:140px;">Letter No.</th>
                <th style="min-width:120px;">Letter Date</th>
                <th style="min-width:140px;">Department</th>
                <th style="min-width:240px;">Subject</th>
                <th style="min-width:120px;">Received</th>
                <th style="min-width:200px;">Scan Copy</th>
                <th style="min-width:140px;">Seen</th>
              </tr>
            </thead>
            <tbody>
            <?php foreach ($letters as $row): ?>
              <tr>
                <td><?= h($row['letter_number']) ?></td>
                <td><?= h($row['letter_date']) ?></td>
                <td><?= h($row['department']) ?></td>
                <td class="text-truncate" style="max-width: 360px;" title="<?= h($row['subject']) ?>">
                    <?= h($row['subject']) ?>
                </td>
                <td><?= $row['received_date'] ? h($row['received_date']) : 'N/A' ?></td>
                <td>
                  <?php if (!empty($row['scan_copy'])): ?>
                    <a class="scan-link" href="<?= h($row['scan_copy']) ?>" target="_blank" rel="noopener">
                      <i class="bi bi-box-arrow-up-right me-1"></i>View Scan
                    </a>
                  <?php else: ?>
                    <span class="text-secondary">No file</span>
                  <?php endif; ?>
                </td>
                <td>
                  <!-- Inline form with a toggle and confirm modal -->
                  <form method="POST" class="seen-form d-inline">
                    <input type="hidden" name="letter_id" value="<?= (int)$row['letter_id'] ?>">
                    <input type="hidden" name="seen" value="1">
                    <button type="button"
                            class="btn btn-sm btn-outline-success d-flex align-items-center gap-1 trigger-confirm"
                            data-letter="<?= h($row['letter_number']) ?>">
                      <i class="bi bi-eye"></i> Mark Seen
                    </button>
                    <input type="hidden" name="update_seen" value="1">
                  </form>
                  <div><span class="badge badge-soft mt-2">Current: No</span></div>
                </td>
              </tr>
            <?php endforeach; ?>
            </tbody>
          </table>
        </div>
      <?php endif; ?>
    </div>
  </main>

  <!-- Confirm Modal -->
  <div class="modal fade" id="confirmSeenModal" tabindex="-1" aria-labelledby="confirmSeenModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered">
      <div class="modal-content">
        <div class="modal-header border-0">
          <h6 class="modal-title" id="confirmSeenModalLabel"><i class="bi bi-question-circle me-2"></i>Confirm Update</h6>
          <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
        </div>
        <div class="modal-body">
          Are you sure you want to mark <strong id="letterRef">this letter</strong> as <span class="text-success">Seen</span>?
        </div>
        <div class="modal-footer border-0">
          <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">Cancel</button>
          <button type="button" class="btn btn-primary" id="confirmSeenBtn"><i class="bi bi-check2-circle me-1"></i>Yes, update</button>
        </div>
      </div>
    </div>
  </div>

  <!-- Bootstrap Bundle (no jQuery needed) -->
  <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
  <script>
    (function(){
      const modalEl = document.getElementById('confirmSeenModal');
      const letterRefEl = document.getElementById('letterRef');
      const confirmBtn = document.getElementById('confirmSeenBtn');
      let pendingForm = null;

      document.querySelectorAll('.trigger-confirm').forEach(btn => {
        btn.addEventListener('click', function(){
          const form = this.closest('form');
          pendingForm = form;
          const ref = this.getAttribute('data-letter') || 'this letter';
          letterRefEl.textContent = ref;
          const modal = new bootstrap.Modal(modalEl);
          modal.show();
        });
      });

      confirmBtn.addEventListener('click', function(){
        if (pendingForm) {
          pendingForm.submit();
        }
      });
    })();
  </script>
</body>
</html>
