<?php
session_start();
require_once 'db_connection.php'; // provides $conn (mysqli) with utf8mb4

// AuthZ
if (!isset($_SESSION['user_id']) || ($_SESSION['role'] ?? '') !== 'branch_head') {
    header("Location: login.php");
    exit();
}

$branch_id = (int)($_SESSION['branch_id'] ?? 0);
if ($branch_id <= 0) {
    header("Location: branch_head_dashboard.php");
    exit();
}

/* ---------- CSRF ---------- */
if (empty($_SESSION['csrf'])) {
    $_SESSION['csrf'] = bin2hex(random_bytes(32));
}
$csrf = $_SESSION['csrf'];
function check_csrf($token) { return hash_equals($_SESSION['csrf'] ?? '', $token); }

/* ---------- Action handlers ---------- */
$success_message = '';
$error_message   = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $token = $_POST['csrf'] ?? '';
    if (!check_csrf($token)) {
        $error_message = "Security check failed. Please retry.";
    } else {
        if (isset($_POST['update_status'])) {
            // Mark as 'received'
            $letter_id = (int)($_POST['letter_id'] ?? 0);
            $new_status = 'received';

            $sql = "UPDATE letters
                    SET status_branch_head = ?, received_date = NOW()
                    WHERE letter_id = ? AND branch_id = ? AND status_branch_head = 'pending'";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("sii", $new_status, $letter_id, $branch_id);
            if ($stmt->execute()) {
                $success_message = "Letter marked as received.";
            } else {
                $error_message = "Error updating status: " . $stmt->error;
            }
            $stmt->close();
        }

        if (isset($_POST['assign_staff'])) {
            // Assign to staff => set status to 'received_assigned', set forward_to and assign_date
            $letter_id = (int)($_POST['letter_id'] ?? 0);
            $forward_to = (int)($_POST['forward_to'] ?? 0);

            if ($forward_to <= 0) {
                $error_message = "Please select a staff member.";
            } else {
                $sql = "UPDATE letters
                        SET forward_to = ?, status_branch_head = 'received_assigned', assign_date = NOW()
                        WHERE letter_id = ? AND branch_id = ?";
                $stmt = $conn->prepare($sql);
                $stmt->bind_param("iii", $forward_to, $letter_id, $branch_id);
                if ($stmt->execute()) {
                    $success_message = "Letter assigned to staff successfully.";
                } else {
                    $error_message = "Error assigning letter: " . $stmt->error;
                }
                $stmt->close();
            }
        }
    }
}

/* ---------- Data fetch ---------- */
// Pending letters for this branch (awaiting branch head)
$sql = "SELECT letter_id, letter_number, letter_date, department, subject,
               status_branch_head, received_date, scan_copy
        FROM letters
        WHERE branch_id = ? AND status_branch_head = 'pending'
        ORDER BY letter_date DESC, letter_id DESC";
$stmt = $conn->prepare($sql);
$stmt->bind_param("i", $branch_id);
$stmt->execute();
$letters = $stmt->get_result();
$stmt->close();

// Staff list (non-branch_head) for this branch (array, not a forward-only cursor)
$staff = [];
$usql = "SELECT user_id, username FROM users
         WHERE branch_id = ? AND role <> 'branch_head' AND status='active'
         ORDER BY username ASC";
$ustmt = $conn->prepare($usql);
$ustmt->bind_param("i", $branch_id);
$ustmt->execute();
$ures = $ustmt->get_result();
while ($u = $ures->fetch_assoc()) { $staff[] = $u; }
$ustmt->close();
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title>Pending Letters – Branch Head</title>

  <!-- Bootstrap 5 -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css"
        rel="stylesheet"
        integrity="sha384-QWTKZyjpPEjISv5WaRU9OFeRpok6YctnYmDr5pNlyT2bRjXh0JMhjY6hW+ALEwIH"
        crossorigin="anonymous" />
  <link rel="stylesheet"
        href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css" />

  <style>
    :root { --brand:#0d6efd; --bg:#f6f8fb; }
    body { background: var(--bg); }
    .page-head {
      background: linear-gradient(135deg, var(--brand), #5aa2ff);
      color: #fff; border-radius: 1rem; padding: 1rem 1.25rem;
      box-shadow: 0 6px 24px rgba(13,110,253,.25);
    }
    .table thead th { background: #fff; position: sticky; top: 0; z-index: 2; }
    .actions { white-space: nowrap; }
    .badge-soft {
      background: rgba(13,110,253,.1);
      border: 1px solid rgba(13,110,253,.2);
      color: #0b1320;
    }
  </style>
</head>
<body>
<div class="container-fluid py-3">
  <div class="page-head mb-3">
    <div class="d-flex align-items-center justify-content-between">
      <div class="d-flex align-items-center gap-3">
        <i class="fa-solid fa-inbox fs-3"></i>
        <div>
          <h5 class="mb-0 fw-bold">Pending Letters</h5>
          <small>Branch: <span class="badge rounded-pill bg-light text-dark">
            <?= htmlspecialchars($_SESSION['branch_id'], ENT_QUOTES, 'UTF-8'); ?></span></small>
        </div>
      </div>
      <div>
        <a href="branch_head_dashboard.php" class="btn btn-light">
          <i class="fa-solid fa-arrow-left-long me-1"></i> Back to Dashboard
        </a>
      </div>
    </div>
  </div>

  <?php if (!empty($success_message)): ?>
    <div class="alert alert-success alert-dismissible fade show" role="alert">
      <?= htmlspecialchars($success_message, ENT_QUOTES, 'UTF-8'); ?>
      <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
  <?php endif; ?>
  <?php if (!empty($error_message)): ?>
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
      <?= htmlspecialchars($error_message, ENT_QUOTES, 'UTF-8'); ?>
      <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
  <?php endif; ?>

  <?php if ($letters->num_rows === 0): ?>
    <div class="card border-0 shadow-sm">
      <div class="card-body text-center py-5">
        <i class="fa-regular fa-circle-check fa-2x mb-2 text-success"></i>
        <p class="mb-0">No pending letters for your branch.</p>
      </div>
    </div>
  <?php else: ?>
    <div class="card border-0 shadow-sm">
      <div class="table-responsive">
        <table class="table align-middle mb-0">
          <thead class="table-light">
            <tr>
              <th style="min-width:140px;">Letter No.</th>
              <th style="min-width:120px;">Date</th>
              <th style="min-width:140px;">Department</th>
              <th>Subject</th>
              <th style="min-width:130px;">Status</th>
              <th style="min-width:130px;">Received Date</th>
              <th style="min-width:110px;">Scan Copy</th>
              <th class="text-end" style="min-width:210px;">Actions</th>
            </tr>
          </thead>
          <tbody>
          <?php while ($row = $letters->fetch_assoc()): ?>
            <tr>
              <td><?= htmlspecialchars($row['letter_number'] ?? '-', ENT_QUOTES, 'UTF-8'); ?></td>
              <td><?= htmlspecialchars($row['letter_date'] ?? '-', ENT_QUOTES, 'UTF-8'); ?></td>
              <td><?= htmlspecialchars($row['department'] ?? '-', ENT_QUOTES, 'UTF-8'); ?></td>
              <td><?= htmlspecialchars($row['subject'] ?? '-', ENT_QUOTES, 'UTF-8'); ?></td>
              <td>
                <span class="badge rounded-pill badge-soft">
                  <?= htmlspecialchars($row['status_branch_head'], ENT_QUOTES, 'UTF-8'); ?>
                </span>
              </td>
              <td><?= !empty($row['received_date']) ? htmlspecialchars($row['received_date'], ENT_QUOTES, 'UTF-8') : '—'; ?></td>
              <td>
                <?php if (!empty($row['scan_copy'])): ?>
                  <a href="<?= htmlspecialchars($row['scan_copy'], ENT_QUOTES, 'UTF-8'); ?>" target="_blank"
                     class="btn btn-sm btn-outline-primary">
                    <i class="fa-solid fa-up-right-from-square me-1"></i> View
                  </a>
                <?php else: ?>
                  <span class="text-muted">None</span>
                <?php endif; ?>
              </td>
              <td class="text-end actions">
                <!-- Mark Received -->
                <form method="post" class="d-inline">
                  <input type="hidden" name="csrf" value="<?= $csrf; ?>">
                  <input type="hidden" name="letter_id" value="<?= (int)$row['letter_id']; ?>">
                  <button type="submit" name="update_status" class="btn btn-sm btn-success">
                    <i class="fa-solid fa-check me-1"></i> Mark Received
                  </button>
                </form>

                <!-- Assign to Staff (modal trigger) -->
                <button type="button"
                        class="btn btn-sm btn-primary ms-1"
                        data-bs-toggle="modal"
                        data-bs-target="#assignModal"
                        data-letter-id="<?= (int)$row['letter_id']; ?>">
                  <i class="fa-solid fa-user-plus me-1"></i> Assign
                </button>
              </td>
            </tr>
          <?php endwhile; ?>
          </tbody>
        </table>
      </div>
    </div>
  <?php endif; ?>
</div>

<!-- Assign Modal -->
<div class="modal fade" id="assignModal" tabindex="-1" aria-labelledby="assignModalLabel" aria-hidden="true">
  <div class="modal-dialog">
    <form method="post" class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="assignModalLabel"><i class="fa-solid fa-user-plus me-2"></i>Assign to Staff</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body">
        <input type="hidden" name="csrf" value="<?= $csrf; ?>">
        <input type="hidden" name="letter_id" id="assignLetterId" value="">
        <div class="mb-3">
          <label for="forward_to" class="form-label">Select Staff</label>
          <select name="forward_to" id="forward_to" class="form-select" required>
            <option value="">— Select —</option>
            <?php foreach ($staff as $u): ?>
              <option value="<?= (int)$u['user_id']; ?>"><?= htmlspecialchars($u['username'], ENT_QUOTES, 'UTF-8'); ?></option>
            <?php endforeach; ?>
          </select>
          <div class="form-text">This will set status to <code>received_assigned</code> and record the assign date.</div>
        </div>
      </div>
      <div class="modal-footer">
        <button class="btn btn-secondary" type="button" data-bs-dismiss="modal">Cancel</button>
        <button class="btn btn-primary" type="submit" name="assign_staff">Assign</button>
      </div>
    </form>
  </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"
        integrity="sha384-YvpcrYf0tY3lHB60NNkmXc5s9fDVZLESaAA55NDzOxhy9GkcIdslK1eN7N6jIeHz"
        crossorigin="anonymous"></script>
<script>
// Pass letter_id to the Assign modal
const assignModal = document.getElementById('assignModal');
assignModal?.addEventListener('show.bs.modal', (event) => {
  const btn = event.relatedTarget;
  const letterId = btn?.getAttribute('data-letter-id') || '';
  document.getElementById('assignLetterId').value = letterId;
});
</script>
</body>
</html>
