<?php
// view_users.php (redesigned)
session_start();
require_once 'db_connection.php';

/* ---------- CSRF token for action links ---------- */
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}
$csrf = $_SESSION['csrf_token'];

/* ---------- Helpers ---------- */
function h($s){ return htmlspecialchars($s ?? '', ENT_QUOTES, 'UTF-8'); }
function selected($a, $b){ return ($a === $b) ? 'selected' : ''; }
function is_valid($v, $allowed){ return in_array($v, $allowed, true); }

/* ---------- Read query params ---------- */
$q         = trim($_GET['q'] ?? '');
$role      = $_GET['role'] ?? '';
$status    = $_GET['status'] ?? '';
$branch_id = $_GET['branch_id'] ?? '';
$sort      = $_GET['sort'] ?? 'user_id';
$dir       = strtolower($_GET['dir'] ?? 'asc');
$page      = max(1, (int)($_GET['page'] ?? 1));
$per_page  = 10;

/* ---------- Whitelists ---------- */
$sortable = ['user_id','username','branch_name','role','status'];
if (!in_array($sort, $sortable, true)) { $sort = 'user_id'; }
$dir = $dir === 'desc' ? 'desc' : 'asc';

$allowed_roles  = ['','letter_admin','branch_head','staff_user','super_admin'];
$allowed_status = ['','active','inactive'];
if (!is_valid($role, $allowed_roles)) $role = '';
if (!is_valid($status, $allowed_status)) $status = '';

/* ---------- Fetch branches for filter ---------- */
$branches = [];
if ($stmt = $conn->prepare("SELECT branch_id, branch_name FROM branches ORDER BY branch_name ASC")) {
    $stmt->execute();
    $res = $stmt->get_result();
    while ($row = $res->fetch_assoc()) { $branches[] = $row; }
    $stmt->close();
}

/* ---------- Build WHERE clauses ---------- */
$where = [];
$params = [];
$types = '';

if ($q !== '') {
    $where[] = "u.username LIKE CONCAT('%', ?, '%')";
    $params[] = $q; $types .= 's';
}
if ($role !== '') {
    $where[] = "u.role = ?";
    $params[] = $role; $types .= 's';
}
if ($status !== '') {
    $where[] = "u.status = ?";
    $params[] = $status; $types .= 's';
}
if ($branch_id !== '') {
    $where[] = "u.branch_id = ?";
    $params[] = $branch_id; $types .= 's';
}
$where_sql = $where ? ('WHERE ' . implode(' AND ', $where)) : '';

/* ---------- Count total ---------- */
$count_sql = "
    SELECT COUNT(*) AS cnt
    FROM users u
    JOIN branches b ON u.branch_id = b.branch_id
    $where_sql
";
$total = 0;
if ($stmt = $conn->prepare($count_sql)) {
    if ($types) { $stmt->bind_param($types, ...$params); }
    $stmt->execute();
    $stmt->bind_result($total);
    $stmt->fetch();
    $stmt->close();
}

/* ---------- Pagination ---------- */
$pages = ($total > 0) ? (int)ceil($total / $per_page) : 1;
if ($page > $pages) $page = $pages;
$offset = ($page - 1) * $per_page;

/* ---------- Data query ---------- */
$data_sql = "
    SELECT u.user_id, u.username, b.branch_name, u.role, u.status
    FROM users u
    JOIN branches b ON u.branch_id = b.branch_id
    $where_sql
    ORDER BY $sort $dir
    LIMIT ? OFFSET ?
";
$rows = [];
if ($stmt = $conn->prepare($data_sql)) {
    if ($types) {
        // add limit + offset
        $bind_types = $types . 'ii';
        $bind_params = array_merge($params, [$per_page, $offset]);
        $stmt->bind_param($bind_types, ...$bind_params);
    } else {
        $stmt->bind_param('ii', $per_page, $offset);
    }
    $stmt->execute();
    $res = $stmt->get_result();
    while ($r = $res->fetch_assoc()) { $rows[] = $r; }
    $stmt->close();
}

$conn->close();

/* ---------- UI helpers ---------- */
function sort_link($label, $key, $curSort, $curDir) {
    $nextDir = ($curSort === $key && $curDir === 'asc') ? 'desc' : 'asc';
    $icon = '';
    if ($curSort === $key) {
        $icon = $curDir === 'asc' ? '↑' : '↓';
    }
    // Preserve existing filters in links
    $params = $_GET; 
    $params['sort'] = $key;
    $params['dir']  = $nextDir;
    $params['page'] = 1;
    $qs = http_build_query($params);
    return '<a class="link-dark text-decoration-none" href="?'.$qs.'">'.h($label).' '.($icon ? '<small>'.$icon.'</small>' : '').'</a>';
}
?>
<!doctype html>
<html lang="en" data-bs-theme="light">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>Users · Office Management System</title>

  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <style>
    body{
      background: #f4f7fb;
      min-height: 100vh;
    }
    .page-wrap{
      max-width: 1200px;
      margin: 0 auto;
      padding: 24px 12px 48px;
    }
    .card{
      border:0; border-radius: 16px;
      box-shadow: 0 10px 24px rgba(0,0,0,.08);
    }
    .role-badge{
      text-transform: capitalize;
    }
  </style>
</head>
<body>
<div class="page-wrap">

  <div class="d-flex align-items-center justify-content-between flex-wrap gap-2 mb-3">
    <a href="super_admin_dashboard.php" class="btn btn-outline-secondary">
      ← Back to Dashboard
    </a>
    <h1 class="h4 m-0">Users</h1>
    <a href="add_user.php" class="btn btn-primary">
      + Add User
    </a>
  </div>

  <!-- Filters -->
  <div class="card mb-3">
    <div class="card-body">
      <form class="row g-3 align-items-end" method="get">
        <div class="col-sm-6 col-lg-3">
          <label class="form-label" for="q">Search (Username)</label>
          <input type="text" id="q" name="q" class="form-control" value="<?= h($q) ?>" placeholder="e.g., kumar">
        </div>
        <div class="col-sm-6 col-lg-3">
          <label class="form-label" for="branch_id">Branch</label>
          <select id="branch_id" name="branch_id" class="form-select">
            <option value="">All Branches</option>
            <?php foreach ($branches as $b): ?>
              <option value="<?= h($b['branch_id']); ?>" <?= selected($branch_id, (string)$b['branch_id']); ?>>
                <?= h($b['branch_name']); ?>
              </option>
            <?php endforeach; ?>
          </select>
        </div>
        <div class="col-sm-6 col-lg-3">
          <label class="form-label" for="role">Role</label>
          <select id="role" name="role" class="form-select">
            <option value="">All Roles</option>
            <option value="letter_admin" <?= selected($role,'letter_admin'); ?>>Letter Admin</option>
            <option value="branch_head"  <?= selected($role,'branch_head');  ?>>Branch Head</option>
            <option value="staff_user"   <?= selected($role,'staff_user');   ?>>Staff User</option>
            <option value="super_admin"  <?= selected($role,'super_admin');  ?>>Super Admin</option>
          </select>
        </div>
        <div class="col-sm-6 col-lg-2">
          <label class="form-label" for="status">Status</label>
          <select id="status" name="status" class="form-select">
            <option value="">All</option>
            <option value="active"   <?= selected($status,'active');   ?>>Active</option>
            <option value="inactive" <?= selected($status,'inactive'); ?>>Inactive</option>
          </select>
        </div>
        <div class="col-sm-6 col-lg-1">
          <button class="btn btn-dark w-100" type="submit">Filter</button>
        </div>
      </form>
    </div>
  </div>

  <!-- Table -->
  <div class="card">
    <div class="table-responsive">
      <table class="table table-hover align-middle mb-0">
        <thead class="table-light">
          <tr>
            <th><?= sort_link('User ID','user_id',$sort,$dir); ?></th>
            <th><?= sort_link('Username','username',$sort,$dir); ?></th>
            <th><?= sort_link('Branch','branch_name',$sort,$dir); ?></th>
            <th><?= sort_link('Role','role',$sort,$dir); ?></th>
            <th><?= sort_link('Status','status',$sort,$dir); ?></th>
            <th class="text-end">Actions</th>
          </tr>
        </thead>
        <tbody>
          <?php if (!$rows): ?>
            <tr><td colspan="6" class="text-center text-muted py-4">No users found</td></tr>
          <?php else: ?>
            <?php foreach ($rows as $row): 
              $nextStatus = ($row['status'] === 'active') ? 'inactive' : 'active';
              $statusLabel = $row['status'] === 'active' ? 'Active' : 'Inactive';
              $statusClass = $row['status'] === 'active' ? 'success' : 'secondary';

              $roleMap = [
                'letter_admin' => 'Letter Admin',
                'branch_head'  => 'Branch Head',
                'staff_user'   => 'Staff User',
                'super_admin'  => 'Super Admin'
              ];
              $roleLabel = $roleMap[$row['role']] ?? $row['role'];

              // Build safe action URLs preserving filters (returns to same page after action if you implement it)
              $baseParams = $_GET; // preserve filters/sort/page
              $baseParams['csrf'] = $csrf;

              $editParams = $baseParams; $editParams['user_id'] = $row['user_id'];
              $statusParams = $baseParams; $statusParams['user_id'] = $row['user_id']; $statusParams['status'] = $nextStatus;

              $editUrl   = 'edit_user.php?'    . http_build_query($editParams);
              $statusUrl = 'change_status.php?'. http_build_query($statusParams);
            ?>
              <tr>
                <td><?= (int)$row['user_id']; ?></td>
                <td><?= h($row['username']); ?></td>
                <td><?= h($row['branch_name']); ?></td>
                <td><span class="badge text-bg-info role-badge"><?= h($roleLabel); ?></span></td>
                <td><span class="badge text-bg-<?= $statusClass; ?>"><?= h($statusLabel); ?></span></td>
                <td class="text-end">
                  <div class="btn-group" role="group">
                    <a href="<?= h($editUrl); ?>" class="btn btn-sm btn-outline-primary"
                       onclick="return confirm('Edit this user?');">
                      Edit
                    </a>
                    <a href="<?= h($statusUrl); ?>" class="btn btn-sm btn-<?= ($row['status']==='active'?'warning':'success'); ?>"
                       onclick="return confirm('Change status to <?= h($nextStatus); ?>?');">
                      <?= ($row['status']==='active'?'Deactivate':'Activate'); ?>
                    </a>
                  </div>
                </td>
              </tr>
            <?php endforeach; ?>
          <?php endif; ?>
        </tbody>
      </table>
    </div>

    <!-- Pagination -->
    <?php if ($pages > 1): 
      // Build page links preserving filters/sort
      $base = $_GET; unset($base['page']);
      $base_qs = http_build_query($base);
      ?>
      <div class="card-body">
        <nav aria-label="User list pages">
          <ul class="pagination justify-content-center m-0">
            <li class="page-item <?= ($page<=1?'disabled':''); ?>">
              <a class="page-link" href="?<?= $base_qs ?>&page=<?= max(1,$page-1) ?>" tabindex="-1">Previous</a>
            </li>
            <?php
              // Compact pager: show first, current±2, last
              $window = 2;
              for ($p=1; $p<=$pages; $p++) {
                if ($p==1 || $p==$pages || ($p>=$page-$window && $p<=$page+$window)) {
                  $active = ($p==$page) ? 'active' : '';
                  echo '<li class="page-item '.$active.'"><a class="page-link" href="?'.$base_qs.'&page='.$p.'">'.$p.'</a></li>';
                } elseif ($p==$page-$window-1 || $p==$page+$window+1) {
                  echo '<li class="page-item disabled"><span class="page-link">…</span></li>';
                }
              }
            ?>
            <li class="page-item <?= ($page>=$pages?'disabled':''); ?>">
              <a class="page-link" href="?<?= $base_qs ?>&page=<?= min($pages,$page+1) ?>">Next</a>
            </li>
          </ul>
        </nav>
      </div>
    <?php endif; ?>
  </div>
</div>

<!-- Bootstrap JS -->
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
